## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>")

## ----installation, eval=FALSE-------------------------------------------------
# if(!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# BiocManager::install("crupR")

## ----setup-for-metadata-------------------------------------------------------
library(crupR)
files <- c(system.file("extdata", "Condition1.H3K4me1.bam", package="crupR"),
           system.file("extdata", "Condition1.H3K4me3.bam", package="crupR"),
           system.file("extdata", "Condition1.H3K27ac.bam", package="crupR"),
           system.file("extdata", "Condition2.H3K4me1.bam", package="crupR"),
           system.file("extdata", "Condition2.H3K4me3.bam", package="crupR"),
           system.file("extdata", "Condition2.H3K27ac.bam", package="crupR"))
                     
inputs <- c(rep(system.file("extdata", "Condition1.Input.bam",
                            package="crupR"), 3), 
            rep(system.file("extdata", "Condition2.Input.bam", 
                            package="crupR"), 3))

## ----metadata-----------------------------------------------------------------
metaData <- data.frame(HM = rep(c("H3K4me1","H3K4me3","H3K27ac"),2),
                       condition = c(1,1,1,2,2,2), replicate = c(1,1,1,1,1,1),
                       bamFile = files, inputFile = inputs)

metaData

## ----run-normalize------------------------------------------------------------
normalized_1 <- normalize(metaData = metaData, condition = 1, replicate = 1, 
                          genome = "mm10", sequencing = "paired")
normalized_2 <- normalize(metaData = metaData, condition = 2, replicate = 1, 
                          genome = "mm10", sequencing = "paired")

## ----show-normalize-----------------------------------------------------------
normalized_1 #the object with the normalized counts
S4Vectors::metadata(normalized_1) #meta data of the samples

## ----show-parameters-normalize------------------------------------------------
normalized_1_inputFree <- normalize(metaData = metaData, condition = 1, 
                                    replicate = 1, genome = "mm10", 
                                    sequencing = "paired", input.free = TRUE)
normalized_1_inputFree

## ----chromsomewise-normalize--------------------------------------------------
normalized_1_chr8 <- normalize(metaData = metaData, condition = 1, 
                               replicate = 1, genome = "mm10", 
                               sequencing = "paired", chroms = c("chr8"))

## ----run-getEnhancers---------------------------------------------------------
prediction_1 <- getEnhancers(data = normalized_1)
prediction_2 <- getEnhancers(data = normalized_2)

## ----run-getEnhancers-ownclassifier, eval = FALSE-----------------------------
# prediction_1_own_class <- getEnhancers(data = normalized_1,
#                                        classifier = "path/to/classifier")

## ----run-getSE----------------------------------------------------------------
se <- getSE(data = prediction_2)
se_strict <- getSE(data = prediction_2, cutoff = 0.9)
se_close <- getSE(data = prediction_2, distance=10000)

## ----list-predictions---------------------------------------------------------
predictions <- list(prediction_1, prediction_2)

## ----run-getDynamics----------------------------------------------------------
clusters <- getDynamics(data = predictions)

## ----show-dynamics------------------------------------------------------------
#clusters
clusters

#meta data
S4Vectors::metadata(clusters)

## ----plotSummary--------------------------------------------------------------
crupR::plotSummary(clusters)

## ----get-expression-----------------------------------------------------------
expression <- readRDS(file = system.file("extdata", "expressions.rds", 
                                         package="crupR"))
expression

## ----run-getTargets-----------------------------------------------------------
targets <- crupR::getTargets(data=clusters, expr = expression, genome = "mm10")

## ----run-getTargets-TADs------------------------------------------------------
path_to_bed <- system.file("extdata", "mESC_mapq30_KR_all_TADs.bed", 
                           package="crupR")
targets <- getTargets(data = clusters, expr = expression, 
                             genome = "mm10", TAD.file = path_to_bed)

## ----run-getTargets-nearest---------------------------------------------------
targets_nearest <- getTargets(data = clusters, expr = NULL, 
                              genome = "mm10", nearest = TRUE)

## ----show-Targets-------------------------------------------------------------
targets

## ----show-saveFiles, eval = FALSE---------------------------------------------
# out_dir <- file.path(tempdir(), 'crupR') #let's use a temporary direcotry for the outputs
# dir.create(out_dir) #create the directory
# #save the GRanges object of the getEnhancers() step
# saveFiles(data = prediction_1, modes = c("bigWig", "rds"), outdir = out_dir)
# #save the GRanges object of the getSE() step
# saveFiles(data = se, modes = c("bedGraph", "bed"), outdir = out_dir)
# #save the GRanges object of the getDynamics() step
# saveFiles(data = clusters, modes = "beds", outdir = out_dir)
# #save the GRanges object of the getTargets() step
# saveFiles(data = targets, modes = "UCSC", outdir = out_dir)

## ----saveFiles-nearest, eval = FALSE------------------------------------------
# saveFiles(data = targets_nearest, modes = "UCSC", outdir = out_dir,
#           nearest = TRUE)

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

