/*
 *
 * Asteroids Player
 *
 * for the c't Competition 2008 (Creativ '08) 
 *
 * Copyright 2008, Volker Raum, Erlangen, Germany
 *
 */
package de.volkerraum.asteroids.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import de.volkerraum.asteroids.configs.ConfigParameters;
import de.volkerraum.asteroids.model.sorter.FutureTargetComparator;
import de.volkerraum.asteroids.model.sorter.InterceptFramesComparator;
import de.volkerraum.asteroids.model.sorter.ReachInterceptAngleComparator;

public class Helper
{
   public static final double TwoPI = Math.PI * 2;
   public static final double internalAngleStep = TwoPI * 3 / 256;

   public static HashMap<Integer, Double> winkelByteToAngle = new HashMap<Integer, Double>();

   public static long count = 0;

   public static boolean isTargetRelevantForShot(BaseObject possibleTarget, double range, AsteroidsModel model)
   {
      boolean isRelevant = true;

      // this object is already hit.. How many Times and what type of object is
      // it ?
      if (!possibleTarget.interceptPossible)
         return false;

      if (possibleTarget instanceof Asteroid)
      {
         Asteroid currAsteroid = (Asteroid) possibleTarget;

         if (model.getAsteroidCount() > 20 && currAsteroid.sizeType != Asteroid.SIZES.SMALL)
         {
            isRelevant = false;
         }

         if (currAsteroid.sizeType == Asteroid.SIZES.SMALL && possibleTarget.shotsFiredAt > 0)
         {
            isRelevant = false; // If a Small One ist Hit => Dead anyway
         }
         else if (currAsteroid.sizeType == Asteroid.SIZES.MEDIUM && possibleTarget.shotsFiredAt > 1)
         {
            isRelevant = false; // If a Small One ist Hit => Dead anyway
         }
         else if (currAsteroid.sizeType == Asteroid.SIZES.BIG && possibleTarget.shotsFiredAt > 2)
         {
            isRelevant = false; // If a Small One ist Hit => Dead anyway
         }
      }
      else if (possibleTarget instanceof Saucer)
      {
         if (possibleTarget.shotsFiredAt > 0)
         {
            isRelevant = false; // Saucers Die immediatly
         }
      }

      // Check for too high Range of asteroids
      if (isRelevant && !(possibleTarget instanceof Saucer) && possibleTarget.interceptDist > range)
      {
         isRelevant = false;
      }
      return isRelevant;
   }

   public static BaseObject calcBestFreeTargetByInterceptAngle(List<BaseObject> objects, Ship ship, double inRange, AsteroidsModel model)
   {
      BaseObject foundObject = null;
      double minAngle = 10000;

      int i = -1;

      for (BaseObject possibleTarget : objects)
      {
         i++;
         if (!isTargetRelevantForShot(possibleTarget, inRange, model))
         {
            continue;
         }

         double shipAngle = ship.angleOfDirection > 0 ? ship.angleOfDirection : TwoPI + ship.angleOfDirection;
         double targetAngle = possibleTarget.interceptAngle > 0 ? possibleTarget.interceptAngle : TwoPI + possibleTarget.interceptAngle;

         double angleDiff = Math.abs(shipAngle - targetAngle);

         if (possibleTarget.distanceToShip < inRange && angleDiff < minAngle)
         {
            minAngle = angleDiff;
            foundObject = possibleTarget;
         }

      }
      return foundObject;
   }

   public static int calcTargetsInSector(List<BaseObject> objects, Ship ship, double angleToCheck, double sectorRange, int framesInTheFuture, double range)
   {
      int targetCount = 0;
      int index = 0;
      for (BaseObject possibleTarget : objects)
      {
         double posX = possibleTarget.x + framesInTheFuture * possibleTarget.dx;
         double posY = possibleTarget.y + framesInTheFuture * possibleTarget.dy;
         double shipPosX = ship.x + framesInTheFuture * ship.dx;
         double shipPosY = ship.y + framesInTheFuture * ship.dy;

         double dist = Helper.calcPythagoras(posX - shipPosX, posY - shipPosY);
         if (dist > range)
            continue;

         double angleToShip = Math.atan2(posY - shipPosY, posX - shipPosX);
         double delta = (angleToShip - angleToCheck);

         if (Math.abs(delta) > Math.PI)
         {
            delta = delta - Math.signum(delta) * Math.PI * 2;
         }

         int oldCount = targetCount;

         if (Math.abs(delta) < sectorRange)
         {
            if (possibleTarget instanceof Asteroid)
            {
               Asteroid asteroid = (Asteroid) possibleTarget;
               if (asteroid.sizeType == Asteroid.SIZES.BIG)
               {
                  targetCount += 3;
               }
               else if (asteroid.sizeType == Asteroid.SIZES.MEDIUM)
               {
                  targetCount += 2;
               }
               else
               {
                  targetCount++;
               }
            }
            else
            {
               targetCount++;
            }
         }
         index++;
      }

      return targetCount;
   }

   public static BaseObject calcBestFreeTargetByInterceptFrames(List<BaseObject> objects, Ship ship, AsteroidsModel model)
   {
      BaseObject foundObject = null;
      int minFrame = 10000;

      for (BaseObject possibleTarget : objects)
      {

         if (!isTargetRelevantForShot(possibleTarget, AsteroidsModel.SHOT_REACH, model))
            continue;

         if (possibleTarget.interceptFrames < minFrame)
         {
            minFrame = possibleTarget.interceptFrames;
            foundObject = possibleTarget;
         }

      }
      return foundObject;
   }

   public static BaseObject calcBestFreeTargetByStepsToReachInterceptAngle(List<BaseObject> objects, Ship ship, AsteroidsModel model)
   {
      BaseObject foundObject = null;
      int minFrame = 10000;

      for (BaseObject possibleTarget : objects)
      {

         if (!isTargetRelevantForShot(possibleTarget, AsteroidsModel.SHOT_REACH, model))
            continue;

         if (possibleTarget.interceptFramesToReachAngle < minFrame)
         {
            minFrame = possibleTarget.interceptFramesToReachAngle;
            foundObject = possibleTarget;
         }

      }
      return foundObject;
   }

   public static BaseObject calcBestFreeTargetByInterceptFramesAndTargetDesity(List<BaseObject> objects, Ship ship, AsteroidsModel model)
   {
      BaseObject foundObject = null;

      List<BaseObject> ratingInterceptFrames = new ArrayList<BaseObject>();
      for (BaseObject currObject : objects)
      {
         if (!isTargetRelevantForShot(currObject, AsteroidsModel.SHOT_REACH, model))
            continue;
         ratingInterceptFrames.add(currObject);
      }
      List<BaseObject> ratingFutureTargetDensity = new ArrayList<BaseObject>(ratingInterceptFrames);

      Collections.sort(ratingInterceptFrames, new InterceptFramesComparator());
      Collections.sort(ratingFutureTargetDensity, new FutureTargetComparator());

      double maxRating = 0;

      for (int i = 0; i < ratingInterceptFrames.size(); ++i)
      {
         BaseObject possibleTarget = ratingInterceptFrames.get(i);

         int indexInFuture = ratingFutureTargetDensity.indexOf(possibleTarget);

         double bigsizeBonus = 0;
         if (possibleTarget instanceof Asteroid)
         {
            Asteroid ast = (Asteroid) possibleTarget;
            if (ast.sizeType == Asteroid.SIZES.BIG)
               bigsizeBonus = ratingFutureTargetDensity.size() / 3;
            else if (ast.sizeType == Asteroid.SIZES.MEDIUM)
               bigsizeBonus = ratingFutureTargetDensity.size() / 5;

         }

         double currRating = ConfigParameters.factorFrames * i + ConfigParameters.factorDensity * indexInFuture + ConfigParameters.factorSize * bigsizeBonus;

         if (currRating > maxRating)
         {
            maxRating = currRating;
            foundObject = ratingInterceptFrames.get(i);
         }
      }

      return foundObject;
   }

   public static BaseObject calcBestFreeTargetByReachInterceptAngleAndTargetDesity(List<BaseObject> objects, Ship ship, AsteroidsModel model)
   {
      BaseObject foundObject = null;

      List<BaseObject> ratingReachAngle = new ArrayList<BaseObject>();
      for (BaseObject currObject : objects)
      {
         if (!isTargetRelevantForShot(currObject, AsteroidsModel.SHOT_REACH, model))
            continue;
         ratingReachAngle.add(currObject);
      }
      List<BaseObject> ratingFutureTargetDensity = new ArrayList<BaseObject>(ratingReachAngle);
      List<BaseObject> ratingInterceptFrames = new ArrayList<BaseObject>(ratingReachAngle);

      Collections.sort(ratingReachAngle, new ReachInterceptAngleComparator());
      Collections.sort(ratingFutureTargetDensity, new FutureTargetComparator());
      Collections.sort(ratingInterceptFrames, new InterceptFramesComparator());

      double maxRating = 0;

      for (int i = 0; i < ratingReachAngle.size(); ++i)
      {
         BaseObject possibleTarget = ratingReachAngle.get(i);

         int indexInFuture = ratingFutureTargetDensity.indexOf(possibleTarget);
         int indexInterceptFrames = ratingInterceptFrames.indexOf(possibleTarget);

         double bigsizeBonus = 0;
         if (possibleTarget instanceof Asteroid)
         {
            Asteroid ast = (Asteroid) possibleTarget;
            if (ast.sizeType == Asteroid.SIZES.BIG)
               bigsizeBonus = ratingFutureTargetDensity.size() / 3;
            else if (ast.sizeType == Asteroid.SIZES.MEDIUM)
               bigsizeBonus = ratingFutureTargetDensity.size() / 5;

         }

         double currRating = 5 * i + 3 * indexInterceptFrames + 1 * indexInFuture;
         // double currRating =
         // ConfigParameters.factorAngleReach * i + ConfigParameters.factorFrames * indexInterceptFrames + ConfigParameters.factorDensity * indexInFuture
         // + ConfigParameters.factorSize * bigsizeBonus;

         if (currRating > maxRating)
         {
            maxRating = currRating;
            foundObject = ratingReachAngle.get(i);
         }
      }

      return foundObject;
   }

   public static BaseObject calcClosestFreeTargetByRange(List<BaseObject> objects, double inRange)
   {
      BaseObject foundObject = null;
      double minRange = 10000;

      for (BaseObject possibleTarget : objects)
      {

         if (possibleTarget.distanceToShip < inRange && possibleTarget.distanceToShip < minRange)
         {
            minRange = possibleTarget.distanceToShip;
            foundObject = possibleTarget;
         }

      }
      return foundObject;
   }

   /**
    * whats the minimum Number any Object was fired at
    * 
    * @param objects
    * @return
    */
   public static int calcMinShotsAtObjects(List<BaseObject> objects)
   {
      int min = Integer.MAX_VALUE;
      for (BaseObject threat : objects)
      {
         if (threat.shotsFiredAt < min)
            min = threat.shotsFiredAt;
      }
      return min;
   }

   public static BaseObject calcBiggestThreat(List<BaseObject> objects)
   {
      Saucer foundSaucer = null;
      BaseObject foundThreat = null;
      int minFrames = 10000;
      for (BaseObject currObject : objects)
      {

         if (currObject.shotsFiredAt > 0 || !currObject.impactImminent || !currObject.interceptPossible)
            continue;

         if (currObject instanceof Saucer)
         {
            foundSaucer = (Saucer) currObject;
         }

         if (currObject.impactFrames < minFrames)
         {
            minFrames = currObject.impactFrames;
            foundThreat = currObject;
         }
      }

      // special handling for the saucer

      if (foundSaucer != null && foundThreat != foundSaucer)
      {
         if (minFrames > 40 && foundSaucer.interceptPossible)
         {
            foundThreat = foundSaucer;

         }

      }

      return foundThreat;
   }

   public static void calcObjectsHitByShots2(List<BaseObject> objects, List<Shot> shots)
   {

      // Loop all Objects
      for (BaseObject currObject : objects)
      {

         // Loop all Shots to check if they hit this Object.
         for (Shot currShot : shots)
         {

            double currShotX;
            double currShotY;
            double currPosX;
            double currPosY;

            for (int step = 0; step < 100; step++)
            {
               // Current position of Object and Shot
               currShotX = currShot.x + currShot.dx * step;
               currShotY = currShot.y + currShot.dy * step;
               currPosX = currObject.x + currObject.dx * step;
               currPosY = currObject.y + currObject.dy * step;

               // Distance of objects
               double dist = Helper.calcPythagoras(currPosX - currShotX, currPosY - currShotY);

               // Radius is important (Size / 2)
               if (dist < currObject.size / 2 + 10)
               {
                  currObject.shotsFiredAt++;
                  break;
               }
            }
         }
      }
   }

   /**
    * Is any Objects interceptPoint in our current direction?
    * 
    * @param objects
    * @param ship
    * @return
    */
   public static BaseObject isInterceptPointInDirection2(List<BaseObject> objects, double futureAngle)
   {
      BaseObject foundInterceptPointForObject = null;

      for (BaseObject currObject : objects)
      {
         if (currObject.shotsFiredAt > 0)
            continue;
         if ((Math.abs(currObject.interceptAngle - futureAngle) < currObject.interceptScatter / 2) && currObject.distanceToShip < 500)
         {
            foundInterceptPointForObject = currObject;
            break;
         }

      }

      return foundInterceptPointForObject;
   }

   public static BaseObject isInterceptPointInDirection(List<BaseObject> objects, int anglebyte)
   {
      BaseObject foundInterceptPointForObject = null;

      for (BaseObject currObject : objects)
      {
         if (currObject.shotsFiredAt > 0)
            continue;
         if (currObject.interceptAngleByte == anglebyte && currObject.distanceToShip < 400)
         {
            foundInterceptPointForObject = currObject;
            break;
         }

      }

      return foundInterceptPointForObject;
   }

   public static double calcDistanceToClosestObject(List<BaseObject> objects, Ship ship)
   {
      double distMin = Double.MAX_VALUE;

      for (BaseObject threat : objects)
      {
         double distToShip = threat.distanceToShip - threat.size / 2 - AsteroidsModel.SHIP_RADIUS; // distance
         // from
         // Center
         // to
         // center
         if (distToShip < distMin)
            distMin = distToShip;
      }

      return distMin;
   }

   public static double getClosestDistanceToAnyObject(List<BaseObject> objects)
   {
      double distMin = Double.MAX_VALUE;

      for (BaseObject threat : objects)
      {
         double newDist = threat.distanceToShip - threat.size / 2 - AsteroidsModel.SHIP_RADIUS;

         if (newDist < distMin)
            distMin = newDist;
      }

      return distMin;
   }

   public static int getFastestImpact(List<BaseObject> objects)
   {
      int impactMin = Integer.MAX_VALUE;

      for (BaseObject threat : objects)
      {
         if (threat.impactImminent && threat.impactFrames < impactMin)
            impactMin = threat.impactFrames;
      }

      return impactMin;
   }

   public static int calcStepsToShotImpact(List<Shot> shots, Ship ship)
   {
      int minStep = 10000;
      for (Shot currShot : shots)
      {
         double lastDistToShot = 0;
         for (int i = 0; i < 500; ++i)
         {
            double shotPosX = currShot.x + currShot.dx * i;
            double shotPosY = currShot.y + currShot.dy * i;
            double shipPosX = ship.x + ship.dx * i;
            double shipPosY = ship.y + ship.dy * i;
            double distToShot = Helper.calcPythagoras(shipPosX - shotPosX, shipPosY - shotPosY);

            if (i > 0)
            {
               if (distToShot > lastDistToShot)
                  break; // shot is moving away
               else
               {

                  if (distToShot < AsteroidsModel.SHIP_SIZE)
                  {

                     if (i < minStep) // Fastest Impact ?
                     {
                        minStep = i;
                     }
                     break; // Next Shot
                  }
               }
            }
            lastDistToShot = distToShot;
         }
      }
      return minStep;
   }

   public static void normalizeObject(Ship ship, BaseObject object)
   {
      if (ship.x - object.x > 512)
         object.x += 1024;
      if (ship.y - object.y > 384)
         object.y += 768;
      if (ship.x - object.x < -512)
         object.x -= 1024;
      if (ship.y - object.y < -384)
         object.y -= 768;
   }

   public static double calcDistance(BaseObject o1, BaseObject o2)
   {
      double dx = o1.x - o2.x;
      double dy = o1.y - o2.y;
      double dist = Math.sqrt(dx * dx + dy * dy);
      return dist;
   }

   public static double calcPythagoras(double a, double b)
   {
      return Math.sqrt(a * a + b * b);
   }

   // can we go the current Direction without being in danger ?
   public static boolean canMoveToCurrentDirection(List<BaseObject> objects, Ship ship)
   {
      double dirX = Math.cos(ship.angleOfDirection) * 1.3;
      double dirY = Math.sin(ship.angleOfDirection) * 1.3;

      boolean canMove = true;
      for (int i = 0; i < 50 && canMove; ++i)
      {
         double posShipX = ship.x + (dirX + ship.dx) * i;
         double posShipY = ship.y + (dirY + ship.dy) * i;

         for (BaseObject currObject : objects)
         {
            double posObjectX = currObject.x + currObject.dx * i;
            double posObjectY = currObject.y + currObject.dy * i;

            double distX = posObjectX - posShipX;
            double distY = posObjectY - posShipY;

            double dist = Helper.calcPythagoras(distX, distY);
            if (dist < (currObject.size / 2 + 15))
            {
               canMove = false;
               break;
            }
         }
      }
      return canMove;
   }

   public static double calcDiffWithTorus(double newValue, double oldValue, int torusSize, int maxDiff)
   {
      double diff = newValue - oldValue;

      // Eliminate Torus Effekt
      if (diff > maxDiff)
      {
         diff = (newValue - torusSize) - oldValue;
      }
      if (diff < -maxDiff)
      {
         diff = (newValue + torusSize) - oldValue;
      }
      return diff;
   }

   public static void calcHighLevelInformations(List<BaseObject> objects, List<Shot> shots, Ship ship, boolean isLeft, boolean isRight)
   {

      for (BaseObject currObject : objects)
      {
         currObject.clearHighLevelValues();
         currObject.distanceToShip = calcPythagoras(ship.x - currObject.x, ship.y - currObject.y);

         currObject.angleToShip = Math.atan2(currObject.y - ship.y, currObject.x - ship.x);

         currObject.shotImpactFrames--;
         if (currObject.shotImpactFrames < 0)
            currObject.shotsFiredAt = 0;
      }

      // calcObjectsHitByShots2(objects, shots);

      double shipX = ship.x;
      double shipY = ship.y;

      for (int step = 1; step < 150; step++)
      {
         // shipX += Math.floor(ship.internalDX) / 8;
         // shipY += Math.floor(ship.internalDY) / 8;

         shipX += ship.dx;
         shipY += ship.dy;

         int asteroidcount = -1;
         for (BaseObject currObject : objects)
         {
            asteroidcount++;

            double objectX = currObject.x + step * currObject.dx;
            double objectY = currObject.y + step * currObject.dy;

            double objectRawX = currObject.rawX + step * currObject.dx;
            double objectRawY = currObject.rawY + step * currObject.dy;

            if (objectRawX < 0)
               objectRawX += 1024;
            if (objectRawX > 1023)
               objectRawX -= 1024;

            if (objectRawY < 0)
               objectRawY += 768;
            if (objectRawY > 767)
               objectRawY -= 768;

            double distToShip = calcPythagoras(shipX - objectX, shipY - objectY);

            // System.err.println("Ast " + asteroidcount + "X/Y " + objectX + "/" + objectY);

            double radius = currObject.size / 2;

            // Impact
            if (!currObject.impactImminent && distToShip < (radius + AsteroidsModel.SHIP_RADIUS + AsteroidsModel.COLLISION_SAFETY))
            {
               currObject.impactFrames = step;
               currObject.impactImminent = true;
               currObject.impactX = objectX;
               currObject.impactY = objectY;
               currObject.impactDist = distToShip - radius - AsteroidsModel.SHIP_RADIUS - AsteroidsModel.COLLISION_SAFETY;
            }

            if (!currObject.interceptPossible)
            {
               // can a shot hit ?
               double angleToIntercept = Math.atan2(objectY - shipY, objectX - shipX);
               double angleToInterceptRaw = Math.atan2(objectRawY - shipY, objectRawX - shipX);

               // this angle may not be reached => Calc the real angle
               double bestFittingToIntercept = findBestFittingAngle(angleToIntercept, ship.angleByte);
               double bestFittingRawToIntercept = findBestFittingAngle(angleToInterceptRaw, ship.angleByte);

               int framesToReachAngle = Math.abs((int) (Helper.calcAngleDifference(bestFittingToIntercept, ship.angleOfDirection) / internalAngleStep));
               int framesToReachAngleRaw = Math.abs((int) (Helper.calcAngleDifference(bestFittingRawToIntercept, ship.angleOfDirection) / internalAngleStep));

               int effectiveShotSteps = step - framesToReachAngle;
               boolean rotateRight = rotateRightForWantedAngle(angleToIntercept, ship.angleByte);

               if (isLeft || isRight)
               {
                  if (rotateRight && isRight)
                     effectiveShotSteps++;
                  else if (rotateRight && !isRight)
                     effectiveShotSteps -= 2;
                  else if (!rotateRight && isLeft)
                     effectiveShotSteps++;
                  else if (!rotateRight && !isLeft)
                     effectiveShotSteps -= 2;
               }

               int effectiveShotStepsRaw = step - framesToReachAngleRaw;
               boolean rotateRightRaw = rotateRightForWantedAngle(angleToInterceptRaw, ship.angleByte);

               if (isLeft || isRight)
               {
                  if (rotateRightRaw && isRight)
                     effectiveShotStepsRaw++;
                  else if (rotateRightRaw && !isRight)
                     effectiveShotStepsRaw -= 2;
                  else if (!rotateRightRaw && isLeft)
                     effectiveShotStepsRaw++;
                  else if (!rotateRightRaw && !isLeft)
                     effectiveShotStepsRaw -= 2;
               }

               if ((effectiveShotStepsRaw < 1) && (effectiveShotSteps < 1))
                  continue;

               double shotDX = Math.cos(bestFittingToIntercept) * AsteroidsModel.SHOT_SPEED;
               double shotDY = Math.sin(bestFittingToIntercept) * AsteroidsModel.SHOT_SPEED;
               shotDX += ship.dx;
               shotDY += ship.dy;
               double shotX = shipX + shotDX * effectiveShotSteps + (0.0 * shotDX); // the first Shot is 20 off the ship
               double shotY = shipY + shotDY * effectiveShotSteps + (0.0 * shotDY);

               // System.err.println("Shot " + shotX + "/" + shotY);

               double shotRawDX = Math.cos(bestFittingRawToIntercept) * AsteroidsModel.SHOT_SPEED;
               double shotRawDY = Math.sin(bestFittingRawToIntercept) * AsteroidsModel.SHOT_SPEED;
               shotRawDX += ship.dx;
               shotRawDY += ship.dy;
               double shotRawX = shipX + shotRawDX * effectiveShotStepsRaw + (0.0 * shotDX);
               double shotRawY = shipY + shotRawDY * effectiveShotStepsRaw + (0.0 * shotDY);

               double interceptDist = calcPythagoras(objectX - shipX, objectY - shipY);
               double interceptDistRaw = calcPythagoras(objectRawX - shipX, objectRawY - shipY);

               // in Range ?
               double distanceOfShotToObject = calcPythagoras(shotX - objectX, shotY - objectY);
               // in Range ?
               double distanceOfShotToObjectRaw = calcPythagoras(shotRawX - objectRawX, shotRawY - objectRawY);

               double rangePenalty = 0;
               if (distanceOfShotToObject > 200)
                  rangePenalty = 1;
               if (distanceOfShotToObject > 400)
                  rangePenalty = 2;

               double rangePenaltyRaw = 0;
               if (distanceOfShotToObjectRaw > 200)
                  rangePenaltyRaw = 1;
               if (distanceOfShotToObjectRaw > 400)
                  rangePenaltyRaw = 2;

               rangePenalty = 0;
               rangePenaltyRaw = 0;

               // System.err.println("Dist " + distanceOfShotToObject + " O X/Y " + objectX + "/" + objectY + " ANGLE " + bestFittingToIntercept);
               // in Range and we can reach the necessary angle in time ?
               if (interceptDist < AsteroidsModel.SHOT_REACH && (distanceOfShotToObject < (radius - rangePenalty)))
               {

                  // System.err.println("Bumm normal " + distanceOfShotToObject);
                  // INTERCEPT
                  currObject.interceptAngle = bestFittingToIntercept;
                  currObject.interceptAngleByte = Helper.findAngleByteForFreeAngle(angleToIntercept, ship.angleByte);

                  currObject.interceptDist = interceptDist;
                  currObject.interceptFrames = step;
                  currObject.interceptX = shotX;
                  currObject.interceptY = shotY;
                  currObject.interceptScatter = Math.asin(radius / currObject.interceptDist);
                  currObject.interceptPossible = true;
                  currObject.interceptFramesToReachAngle = framesToReachAngle;
                  currObject.futureTargetCountInSector = calcTargetsInSector(objects, ship, currObject.interceptAngle, 0.52, step, 400);

               }

               // in Range and we can reach the necessary angle in time ?
               else if (interceptDistRaw < AsteroidsModel.SHOT_REACH && (distanceOfShotToObjectRaw < (radius - rangePenaltyRaw)))
               {
                  // System.err.println("Bumm raw " + distanceOfShotToObjectRaw);
                  // INTERCEPT
                  currObject.interceptAngle = bestFittingRawToIntercept;
                  currObject.interceptAngleByte = Helper.findAngleByteForFreeAngle(angleToInterceptRaw, ship.angleByte);

                  currObject.interceptDist = interceptDistRaw;
                  currObject.interceptFrames = step;
                  currObject.interceptX = shotRawX;
                  currObject.interceptY = shotRawY;
                  currObject.interceptScatter = Math.asin(radius / currObject.interceptDist);
                  currObject.interceptPossible = true;
                  currObject.interceptFramesToReachAngle = framesToReachAngleRaw;
                  currObject.futureTargetCountInSector = calcTargetsInSector(objects, ship, currObject.interceptAngle, 0.52, step, 400);

               }

            }
         }
      }

   }

   public static void calcInterceptPointsForObjects(List<BaseObject> objects, Ship ship, int framesIntoFuture)
   {

      int objectCount = -1;
      for (BaseObject currObject : objects)
      {
         currObject.interceptPoints.clear();
         objectCount++;
         double shipX = ship.x;
         double shipY = ship.y;

         double objectX = currObject.rawX;
         double objectY = currObject.rawY;
         double radius = currObject.size / 2;

         for (int frame = 1; frame < framesIntoFuture; frame++)
         {
            objectX += currObject.dx;
            objectY += currObject.dy;

            double normalizedX = objectX;
            double normalizedY = objectY;

            // correct the Raw Position
            if (normalizedX < 0)
               normalizedX += 1024;

            if (normalizedY < 0)
               normalizedY += 768;

            if (normalizedX > 1023)
               normalizedX -= 1024;

            if (normalizedY > 767)
               normalizedY -= 768;

            // NOW THE TORUS STUFF

            if (ship.x - normalizedX > 512)
               normalizedX += 1024;

            if (ship.y - normalizedY > 384)
               normalizedY += 768;

            if (ship.x - normalizedX < -512)
               normalizedX -= 1024;

            if (ship.y - normalizedY < -384)
               normalizedY -= 768;

            double angleToIntercept = Math.atan2(normalizedY - shipY, normalizedX - shipX);
            double distToShip = calcPythagoras(shipX - normalizedX, shipY - normalizedY);
            // if we shoot at the current position => the target must be n frames away from that
            int framesToPast = (int) (distToShip / AsteroidsModel.SHOT_SPEED);

            if (distToShip < 560 && (frame - framesToPast) > 0) // shots only fly about 560 pixel ;
            {

               InterceptPoint newPoint = new InterceptPoint();
               newPoint.x = objectX;
               newPoint.y = objectY;
               newPoint.angle = angleToIntercept;
               newPoint.dist = distToShip;
               newPoint.frame = frame - framesToPast;
               newPoint.framesForShotTravel = (int) (distToShip / 8);
               newPoint.scatter = Math.asin(radius / currObject.interceptDist);
               currObject.interceptPoints.add(newPoint);
               currObject.frameToInterceptPoint.put(newPoint.frame, newPoint);
            }
         }
      }
   }

   public static BaseObject calcBestTargetByLongTermAnalysis(List<BaseObject> objects, Ship ship, AsteroidsModel theModel, int recursion)
   {
      double minRatio = 1000000;
      BaseObject bestObject = null;
      ArrayList<Integer> usedObjects = null;

      double ratio = 0;

      for (BaseObject currObject : objects)
      {
         if (currObject.interceptPossible && Helper.isTargetRelevantForShot(currObject, 560, theModel))
         {
            usedObjects = new ArrayList<Integer>();
            int frames = bestIntercept(objects, ship.internalAngle, 0, usedObjects, recursion);

            ratio = ((double) frames) / usedObjects.size();

            if (ratio < minRatio)
            {
               minRatio = ratio;
               bestObject = currObject;
            }
         }
      }
      return bestObject;
   }

   private static int bestIntercept(List<BaseObject> objects, double startAngle, int currFrame, List<Integer> usedObjectIndices, int recursionDepths)
   {
      count++;
      int currFrameCount = currFrame;

      double minAngleDiff = 1000;
      int framesToReachMinAngle = 0;
      int objectIndexForMinAngle = 0;
      boolean foundIntercept = false;
      double interceptAngle = 0;
      for (int i = 0; i < objects.size(); ++i)
      {
         if (!usedObjectIndices.contains(i))
         {
            BaseObject currObject = (BaseObject) objects.get(i);
            int frameCount = currFrame;
            InterceptPoint point = null;
            while (frameCount < currObject.interceptPoints.size())
            {
               point = currObject.frameToInterceptPoint.get(frameCount);
               if (point != null)
               {
                  break;
               }
               frameCount++;
            }
            if (point == null)
               break;

            double angleDiff = Math.abs(Helper.calcAngleDifference(startAngle, point.angle));
            if (minAngleDiff > angleDiff)
            {
               minAngleDiff = angleDiff;
               framesToReachMinAngle = (int) (Math.abs(angleDiff) / internalAngleStep);
               objectIndexForMinAngle = i;
               foundIntercept = true;
               interceptAngle = point.angle;
            }
         }
      }
      if (foundIntercept)
      {
         usedObjectIndices.add(objectIndexForMinAngle);
         currFrameCount += framesToReachMinAngle;
      }

      if (recursionDepths > 1 && foundIntercept)
      {

         currFrameCount += bestIntercept(objects, interceptAngle, currFrameCount, usedObjectIndices, recursionDepths - 1);
      }

      return currFrameCount;
   }

   public static double findBestFittingAngle(double wantedAngle, int currentAngleByte)
   {
      return winkelByteToAngle.get(findAngleByteForFreeAngle(wantedAngle, currentAngleByte));
   }

   public static int findAngleByteForFreeAngle(double wantedAngle, int currentAngleByte)
   {
      double currAngleByAngleByte = winkelByteToAngle.get(currentAngleByte);
      double diff = wantedAngle - currAngleByAngleByte;

      if (diff > Math.PI)
      {
         diff = diff - TwoPI;
      }
      if (diff < -Math.PI)
      {
         diff = diff + TwoPI;
      }

      int steps = (int) Math.round(diff / internalAngleStep) * 3; // rotations
      // are in
      // steps by 3
      // AngleByte

      int newAngleByte = currentAngleByte + steps;
      if (newAngleByte < 0)
         newAngleByte += 256;
      if (newAngleByte > 255)
         newAngleByte -= 256;

      return newAngleByte;

   }

   public static boolean rotateRightForWantedAngle(double wantedAngle, int currentAngleByte)
   {
      double currAngleByAngleByte = winkelByteToAngle.get(currentAngleByte);
      double diff = wantedAngle - currAngleByAngleByte;

      if (diff > Math.PI)
      {
         diff = diff - TwoPI;
      }
      if (diff < -Math.PI)
      {
         diff = diff + TwoPI;
      }

      int steps = (int) Math.round(diff / internalAngleStep) * 3; // rotations
      // are in
      // steps by 3
      // AngleByte
      return steps < 0;

   }

   public static double roundToDigits(double number, double digits)
   {
      return ((double) Math.round(number * digits)) / digits;
   }

   public static double getAngleForAngleByte(int angleByte)
   {
      return winkelByteToAngle.get(angleByte);
   }

   public static double calcAngleDifference(double angle1, double angle2)
   {
      double diff = angle2 - angle1;
      if (diff > Math.PI)
      {
         diff = diff - TwoPI;
      }
      if (diff < -Math.PI)
      {
         diff = diff + TwoPI;
      }
      return diff;
   }

   public static void main(String[] args)
   {
      BaseObject bo = new BaseObject();
      bo.x = -3;
      bo.y = -0.2;
      Ship testShip = new Ship();
      testShip.internalAngle = -0.14;

      List<BaseObject> objects = new ArrayList<BaseObject>();
      objects.add(bo);

      calcTargetsInSector(objects, testShip, 0.4, 0.3, 0, 400);
   }

   static
   {
      winkelByteToAngle.put(0, 0.0);
      winkelByteToAngle.put(3, 0.06283185307179587);
      winkelByteToAngle.put(6, 0.143116998663535);
      winkelByteToAngle.put(9, 0.22165681500327983);
      winkelByteToAngle.put(12, 0.2879793265790644);
      winkelByteToAngle.put(15, 0.3700098014227978);
      winkelByteToAngle.put(18, 0.439822971502571);
      winkelByteToAngle.put(21, 0.5148721293383272);
      winkelByteToAngle.put(24, 0.5846852994181003);
      winkelByteToAngle.put(27, 0.6597344572538565);
      winkelByteToAngle.put(30, 0.7278022980816355);
      winkelByteToAngle.put(33, 0.8063421144213803);
      winkelByteToAngle.put(36, 0.8866272600131193);
      winkelByteToAngle.put(39, 0.9651670763528643);
      winkelByteToAngle.put(42, 1.0367255756846316);
      winkelByteToAngle.put(45, 1.106538745764405);
      winkelByteToAngle.put(48, 1.1780972450961724);
      winkelByteToAngle.put(51, 1.24965574442794);
      winkelByteToAngle.put(54, 1.3281955607676845);
      winkelByteToAngle.put(57, 1.3945180723434694);
      winkelByteToAngle.put(60, 1.4748032179352084);
      winkelByteToAngle.put(63, 1.5550883635269477);
      winkelByteToAngle.put(66, 1.6196655458507379);
      winkelByteToAngle.put(69, 1.6964600329384885);
      winkelByteToAngle.put(72, 1.7767451785302275);
      winkelByteToAngle.put(75, 1.8413223608540177);
      winkelByteToAngle.put(78, 1.9233528356977512);
      winkelByteToAngle.put(81, 1.9914206765255298);
      winkelByteToAngle.put(84, 2.061233846605303);
      winkelByteToAngle.put(87, 2.1432643214490366);
      winkelByteToAngle.put(90, 2.2113321622768156);
      winkelByteToAngle.put(93, 2.288126649364566);
      winkelByteToAngle.put(96, 2.356194490192345);
      winkelByteToAngle.put(99, 2.4329889772800954);
      winkelByteToAngle.put(102, 2.5010568181078745);
      winkelByteToAngle.put(105, 2.5813419636996136);
      winkelByteToAngle.put(108, 2.651155133779387);
      winkelByteToAngle.put(111, 2.7331856086231197);
      winkelByteToAngle.put(114, 2.804744107954887);
      winkelByteToAngle.put(117, 2.871066619530672);
      winkelByteToAngle.put(120, 2.9530970943744057);
      winkelByteToAngle.put(123, 3.015928947446201);
      winkelByteToAngle.put(126, 3.0944687637859465);
      winkelByteToAngle.put(129, -3.1084313978019007);
      winkelByteToAngle.put(132, -3.0455995447301047);
      winkelByteToAngle.put(135, -2.968805057642354);
      winkelByteToAngle.put(138, -2.888519912050615);
      winkelByteToAngle.put(141, -2.823942729726826);
      winkelByteToAngle.put(144, -2.7401669256310974);
      winkelByteToAngle.put(147, -2.672099084803319);
      winkelByteToAngle.put(150, -2.600540585471551);
      winkelByteToAngle.put(153, -2.5237460983838003);
      winkelByteToAngle.put(156, -2.443460952792061);
      winkelByteToAngle.put(159, -2.377138441216277);
      winkelByteToAngle.put(162, -2.3125612588924866);
      winkelByteToAngle.put(165, -2.230530784048753);
      winkelByteToAngle.put(168, -2.165953601724963);
      winkelByteToAngle.put(171, -2.089159114637212);
      winkelByteToAngle.put(174, -2.005383310541484);
      winkelByteToAngle.put(177, -1.9408061282176945);
      winkelByteToAngle.put(180, -1.8727382873899154);
      winkelByteToAngle.put(183, -1.792453141798176);
      winkelByteToAngle.put(186, -1.726130630222392);
      winkelByteToAngle.put(189, -1.6475908138826467);
      winkelByteToAngle.put(192, -1.5707963267948966);
      winkelByteToAngle.put(195, -1.5079644737231002);
      winkelByteToAngle.put(198, -1.4276793281313616);
      winkelByteToAngle.put(201, -1.3491395117916172);
      winkelByteToAngle.put(204, -1.2828170002158323);
      winkelByteToAngle.put(207, -1.2007865253720986);
      winkelByteToAngle.put(210, -1.1344640137963138);
      winkelByteToAngle.put(213, -1.0646508437165414);
      winkelByteToAngle.put(216, -0.9861110273767961);
      winkelByteToAngle.put(219, -0.9110618695410402);
      winkelByteToAngle.put(222, -0.8429940287132611);
      winkelByteToAngle.put(225, -0.7644542123735159);
      winkelByteToAngle.put(228, -0.6981317007977319);
      winkelByteToAngle.put(231, -0.6195918844579866);
      winkelByteToAngle.put(234, -0.5480333851262191);
      winkelByteToAngle.put(237, -0.47647488579445163);
      winkelByteToAngle.put(240, -0.40840704496667346);
      winkelByteToAngle.put(243, -0.3211405823669562);
      winkelByteToAngle.put(246, -0.25656340004316647);
      winkelByteToAngle.put(249, -0.17627825445142786);
      winkelByteToAngle.put(252, -0.09599310885968837);
      winkelByteToAngle.put(255, -0.033161255787891974);
      winkelByteToAngle.put(2, 0.048869219055841226);
      winkelByteToAngle.put(5, 0.12566370614359174);
      winkelByteToAngle.put(8, 0.19198621771937624);
      winkelByteToAngle.put(11, 0.27052603405912107);
      winkelByteToAngle.put(14, 0.3368485456349056);
      winkelByteToAngle.put(17, 0.40840704496667307);
      winkelByteToAngle.put(20, 0.4904375198104066);
      winkelByteToAngle.put(23, 0.57246799465414);
      winkelByteToAngle.put(26, 0.640535835481919);
      winkelByteToAngle.put(29, 0.7086036763096979);
      winkelByteToAngle.put(32, 0.7853981633974483);
      winkelByteToAngle.put(35, 0.8621926504851988);
      winkelByteToAngle.put(38, 0.9302604913129776);
      winkelByteToAngle.put(41, 0.9983283321407566);
      winkelByteToAngle.put(44, 1.08035880698449);
      winkelByteToAngle.put(47, 1.1623892818282233);
      winkelByteToAngle.put(50, 1.233947781159991);
      winkelByteToAngle.put(53, 1.3002702927357754);
      winkelByteToAngle.put(56, 1.3788101090755203);
      winkelByteToAngle.put(59, 1.4451326206513049);
      winkelByteToAngle.put(62, 1.5219271077390555);
      winkelByteToAngle.put(65, 1.6039575825827888);
      winkelByteToAngle.put(68, 1.6667894356545847);
      winkelByteToAngle.put(71, 1.7470745812463238);
      winkelByteToAngle.put(74, 1.827359726838063);
      winkelByteToAngle.put(77, 1.8919369091618532);
      winkelByteToAngle.put(80, 1.9792033717615696);
      winkelByteToAngle.put(83, 2.047271212589348);
      winkelByteToAngle.put(86, 2.118829711921116);
      winkelByteToAngle.put(89, 2.1921335405048774);
      winkelByteToAngle.put(92, 2.2689280275926285);
      winkelByteToAngle.put(95, 2.335250539168413);
      winkelByteToAngle.put(98, 2.4137903555081577);
      winkelByteToAngle.put(101, 2.4818581963359363);
      winkelByteToAngle.put(104, 2.5569073541716927);
      winkelByteToAngle.put(107, 2.6354471705114375);
      winkelByteToAngle.put(110, 2.705260340591211);
      winkelByteToAngle.put(113, 2.7715828521669956);
      winkelByteToAngle.put(116, 2.855358656262723);
      winkelByteToAngle.put(119, 2.9199358385865133);
      winkelByteToAngle.put(122, 2.998475654926258);
      winkelByteToAngle.put(125, 3.0822514590219856);
      winkelByteToAngle.put(128, 3.141592653589793);
      winkelByteToAngle.put(131, -3.0647981665020425);
      winkelByteToAngle.put(134, -2.9862583501622977);
      winkelByteToAngle.put(137, -2.9199358385865137);
      winkelByteToAngle.put(140, -2.839650692994774);
      winkelByteToAngle.put(143, -2.771582852166995);
      winkelByteToAngle.put(146, -2.705260340591211);
      winkelByteToAngle.put(149, -2.6232298657474775);
      winkelByteToAngle.put(152, -2.5446900494077327);
      winkelByteToAngle.put(155, -2.481858196335936);
      winkelByteToAngle.put(158, -2.3998277214922035);
      winkelByteToAngle.put(161, -2.3352505391684133);
      winkelByteToAngle.put(164, -2.2689280275926285);
      winkelByteToAngle.put(167, -2.18864288200089);
      winkelByteToAngle.put(170, -2.111848394913139);
      winkelByteToAngle.put(173, -2.0402898955813713);
      winkelByteToAngle.put(176, -1.9722220547535922);
      winkelByteToAngle.put(179, -1.8884462506578643);
      winkelByteToAngle.put(182, -1.8238690683340746);
      winkelByteToAngle.put(185, -1.743583922742335);
      winkelByteToAngle.put(188, -1.666789435654585);
      winkelByteToAngle.put(191, -1.6039575825827885);
      winkelByteToAngle.put(194, -1.52367243699105);
      winkelByteToAngle.put(197, -1.4451326206513047);
      winkelByteToAngle.put(200, -1.3823007675795091);
      winkelByteToAngle.put(203, -1.3002702927357754);
      winkelByteToAngle.put(206, -1.2339477811599906);
      winkelByteToAngle.put(209, -1.162389281828224);
      winkelByteToAngle.put(212, -1.0803588069844894);
      winkelByteToAngle.put(215, -1.0088003076527228);
      winkelByteToAngle.put(218, -0.9302604913129775);
      winkelByteToAngle.put(221, -0.8621926504851984);
      winkelByteToAngle.put(224, -0.7853981633974483);
      winkelByteToAngle.put(227, -0.7173303225696701);
      winkelByteToAngle.put(230, -0.6405358354819191);
      winkelByteToAngle.put(233, -0.57246799465414);
      winkelByteToAngle.put(236, -0.4904375198104072);
      winkelByteToAngle.put(239, -0.4206243497306339);
      winkelByteToAngle.put(242, -0.35255650890285395);
      winkelByteToAngle.put(245, -0.270526034059122);
      winkelByteToAngle.put(248, -0.2059488517353314);
      winkelByteToAngle.put(251, -0.1256637061435919);
      winkelByteToAngle.put(254, -0.04886921905584174);
      winkelByteToAngle.put(1, 0.015707963267948967);
      winkelByteToAngle.put(4, 0.09599310885968812);
      winkelByteToAngle.put(7, 0.17627825445142728);
      winkelByteToAngle.put(10, 0.24260076602721178);
      winkelByteToAngle.put(13, 0.3211405823669566);
      winkelByteToAngle.put(16, 0.39269908169872414);
      winkelByteToAngle.put(19, 0.4642575810304917);
      winkelByteToAngle.put(22, 0.5340707511102648);
      winkelByteToAngle.put(25, 0.6056292504420324);
      winkelByteToAngle.put(28, 0.6841690667817772);
      winkelByteToAngle.put(31, 0.7644542123735162);
      winkelByteToAngle.put(34, 0.8429940287132611);
      winkelByteToAngle.put(37, 0.9110618695410401);
      winkelByteToAngle.put(40, 0.9861110273767961);
      winkelByteToAngle.put(43, 1.0559241974565694);
      winkelByteToAngle.put(46, 1.1292280260403311);
      winkelByteToAngle.put(49, 1.2007865253720986);
      winkelByteToAngle.put(52, 1.2828170002158323);
      winkelByteToAngle.put(55, 1.3491395117916167);
      winkelByteToAngle.put(58, 1.4276793281313616);
      winkelByteToAngle.put(61, 1.5079644737231006);
      winkelByteToAngle.put(64, 1.5707963267948966);
      winkelByteToAngle.put(67, 1.6493361431346414);
      winkelByteToAngle.put(70, 1.726130630222392);
      winkelByteToAngle.put(73, 1.7924531417981766);
      winkelByteToAngle.put(76, 1.8727382873899157);
      winkelByteToAngle.put(79, 1.9408061282176943);
      winkelByteToAngle.put(82, 2.012364627549462);
      winkelByteToAngle.put(85, 2.097885760897184);
      winkelByteToAngle.put(88, 2.1676989309769574);
      winkelByteToAngle.put(91, 2.230530784048753);
      winkelByteToAngle.put(94, 2.3125612588924866);
      winkelByteToAngle.put(97, 2.375393111964282);
      winkelByteToAngle.put(100, 2.4574235868080163);
      winkelByteToAngle.put(103, 2.5359634031477607);
      winkelByteToAngle.put(106, 2.6145032194875055);
      winkelByteToAngle.put(109, 2.686061718819273);
      winkelByteToAngle.put(112, 2.7558748888990463);
      winkelByteToAngle.put(115, 2.8239427297268254);
      winkelByteToAngle.put(118, 2.90248254606657);
      winkelByteToAngle.put(121, 2.9688050576423546);
      winkelByteToAngle.put(124, 3.0421088862261167);
      winkelByteToAngle.put(127, 3.1258846903218442);
      winkelByteToAngle.put(130, -3.0909781052819576);
      winkelByteToAngle.put(133, -3.0159289474462017);
      winkelByteToAngle.put(136, -2.9408797896104457);
      winkelByteToAngle.put(139, -2.8693212902786778);
      winkelByteToAngle.put(142, -2.787290815434944);
      winkelByteToAngle.put(145, -2.7209683038591597);
      winkelByteToAngle.put(148, -2.651155133779387);
      winkelByteToAngle.put(151, -2.5813419636996136);
      winkelByteToAngle.put(154, -2.5028021473598683);
      winkelByteToAngle.put(157, -2.4260076602721177);
      winkelByteToAngle.put(160, -2.356194490192345);
      winkelByteToAngle.put(163, -2.288126649364566);
      winkelByteToAngle.put(166, -2.2113321622768156);
      winkelByteToAngle.put(169, -2.1327923459370703);
      winkelByteToAngle.put(172, -2.061233846605303);
      winkelByteToAngle.put(175, -1.9914206765255296);
      winkelByteToAngle.put(178, -1.9233528356977505);
      winkelByteToAngle.put(181, -1.8413223608540177);
      winkelByteToAngle.put(184, -1.7732545200262395);
      winkelByteToAngle.put(187, -1.6964600329384876);
      winkelByteToAngle.put(190, -1.6179202165987432);
      winkelByteToAngle.put(193, -1.5550883635269477);
      winkelByteToAngle.put(196, -1.474803217935209);
      winkelByteToAngle.put(199, -1.398008730847458);
      winkelByteToAngle.put(202, -1.3316862192716732);
      winkelByteToAngle.put(205, -1.2531464029319288);
      winkelByteToAngle.put(208, -1.1850785621041497);
      winkelByteToAngle.put(211, -1.1152653920243765);
      winkelByteToAngle.put(214, -1.043706892692609);
      winkelByteToAngle.put(217, -0.9651670763528646);
      winkelByteToAngle.put(220, -0.8866272600131193);
      winkelByteToAngle.put(223, -0.8063421144213798);
      winkelByteToAngle.put(226, -0.7417649320975901);
      winkelByteToAngle.put(229, -0.6597344572538564);
      winkelByteToAngle.put(232, -0.59690260418206);
      winkelByteToAngle.put(235, -0.5270894341022867);
      winkelByteToAngle.put(238, -0.44156830075456543);
      winkelByteToAngle.put(241, -0.37000980142279793);
      winkelByteToAngle.put(244, -0.30194196059501976);
      winkelByteToAngle.put(247, -0.2234021442552736);
      winkelByteToAngle.put(250, -0.15533430342749543);
      winkelByteToAngle.put(253, -0.07853981633974527);

   }
}
